#!/bin/bash

# Copyright (c) 2020 by Pulse Secure, LLC. All rights reserved

HOMEDIR=$HOME
INSTALLDIR=/usr/local/pulse
PULSEDIR=$HOME/.pulsesecure/pulse
PULSECERTDIR=$PULSEDIR/certificates
PRIVATESUBDIR=.private
args=""
ive_ip=""
NOARGS=$#
SCRARGS=$@
OPENSSLCMD=openssl
CERTUTIL=/usr/bin/certutil
PK12UTIL=/usr/bin/pk12util

SCRNAME=`basename $0`

function check_error()
{
	errorCode=$1
	errorString=$2
	if [ $1 != 0 ] && [ "X$errorString" != "X" ]; then 
		echo "ErrorMessage : $errorString" 
		exit 3
	fi
}

function install_pfx()
{
	filename=$1
	pfxFilepath=$(readlink -f "$filename")
	keyFileBaseName=$(basename "$filename")
	keyFileName="${keyFileBaseName%.*}"
	privKeyFileName="$PULSECERTDIR/$PRIVATESUBDIR/$keyFileName.key"
	pubKeyFileName="$PULSECERTDIR/$keyFileName.pem"
	pubKeyTmpFileName="$PULSECERTDIR/$keyFileName-tmp-pub.pem"

	# pkcs12 file format support starts here
	if [ ! -f "$filename" ]; then 
		echo "$filename does not exists. Please check the pfx file location "
		exit 2;
	fi

    warn_user_for_overwrite "${pubKeyFileName}" 
	#$OPENSSLCMD pkcs12 -info -in $filename -passin pass:$password -nodes 2>/dev/null
	#$OPENSSLCMD pkcs12 -info -in $filename -nodes 2>/dev/null
	#check_error $? "$FUNCNAME: File Extension is .pfx/.p12 but content is not"
	echo "Certificates will be installed for Pulse Certificate store and CEF Certificate store."
    echo -n Enter Import Password:
    	read -s paswd

	echo "Extracting Public Key from $filename"
	pubKeyExtractCmd='$OPENSSLCMD pkcs12 -in '\"$filename\"' -clcerts -nokeys -out '\"$pubKeyTmpFileName\"' -nodes -passin pass:'\"$paswd\"' -passout pass:'
	eval $pubKeyExtractCmd
	ret=$?
	if [ $ret != 0 ]; then
		if [ -e $pubKeyTmpFileName ]
		then
			rm "$pubKeyTmpFileName"
		fi
	fi
	check_error $ret "$FUNCNAME: Public key extraction failed"

	check_already_installed "${pubKeyTmpFileName}"
	mv $pubKeyTmpFileName $pubKeyFileName 

	echo "Extracting Private Key from $filename"
	privKeyExtractCmd='$OPENSSLCMD pkcs12 -in '\"$filename\"' -nocerts -out '\"$privKeyFileName\"' -nodes  -passin pass:'\"$paswd\"' -passout pass:'
	eval $privKeyExtractCmd
	ret=$?
	if [ $ret != 0 ]; then
		rm "$privKeyFileName"
		rm "$pubKeyFileName"
	fi
	check_error $ret "$FUNCNAME: Private key extraction failed"
    echo "Successfully added certificate to Pulse Certificate store."
    
    pk12CMD='${PK12UTIL} -d sql:$HOME/.pki/nssdb  -i '\"${filename}\"'  -W '\"${paswd}\"
    eval $pk12CMD
    ret=$?
    if [ $ret != 0 ]; then
        echo "Failed to add certificate to CEF Certificate store."
    else 
        echo "Successfully added certificate to CEF Certificate store."
    fi
}

function warn_user_for_overwrite()
{
    certFile=$1
    if [ -f "$certFile" ]; then
        name=$(basename "$certFile" ".pem")
        echo 
        echo "Client certificate with name ${name} already"\
             "exists in pulse certificate store."
        read -e -p "Do you want to continue[y/n]: " choice
        if ! [[ "${choice:0:1}" == "Y" || "${choice:0:1}" == "y" ]]; then
            echo "Aborting the certificate installation."
            exit 0;
        fi
    fi
}

function check_already_installed()
{
	certFile=$1
	if ls $PULSECERTDIR/*.pem &>/dev/null
	then
		opensslCNCmd='$OPENSSLCMD x509 -noout -subject_hash -in '\"$certFile\"''
		CNHashOld=$(eval $opensslCNCmd 2>&1 )
   		for i in $PULSECERTDIR/*.pem;
		do
			# skip checking against the same file 
			if [ $certFile != $i ]; then
			openHashNew='$OPENSSLCMD x509 -noout -subject_hash -in '\"$i\"''
				CNHashNew=$(eval $openHashNew 2>&1 )
				if [ "$CNHashOld" = "$CNHashNew" ]; then
					echo "Certificate is already present in pulse certificate store. Aborting the certificate installation."
					rm $certFile
					exit 0;
				fi
			fi
		done
	fi
}

function check_cert_names_same()
{
    priv="$1"
    pub="$2"
    priv=$(basename "$priv")
    priv=${priv%.*}
    pub=$(basename "$pub")
    pub=${pub%.*}
    if [ "$priv" != "$pub" ]; then
        echo "Failed to install certificate. Both Private($priv) and Public($pub) certificate should have same name."
        exit 0;
    fi
}

function install_keys()
{
	FILETYPE=$1
	privKeyInFile="$2"
	privKeyOutFile="$3"
	FAIL=1
	keytype="rsa dsa"
	for i in `echo $keytype`
	do
		installKeyCmd='$OPENSSLCMD '$i' -inform '$FILETYPE' -in '\"$privKeyInFile\"' -out '\"$privKeyOutFile\"' 2>/dev/null'
		eval $installKeyCmd
		if [ $? == 0 ]; then
			FAIL=0
			break;
		fi
	done
	check_error $FAIL "Failed to extract private keys. Supported keys are rsa and dsa"
}

function install_priv_pub_keys()
{
	privKeyFilePath=$1
	privKeyFileBaseName=$(basename "$1")
	privKeyFileName="${privKeyFileBaseName%.*}"
	privKeyFileExt="${privKeyFileBaseName##*.}"
	pubKeyFilePath=$2
	pubKeyFileBaseName=$(basename "$2")
	pubKeyFileName="${pubKeyFileBaseName%.*}"
	pubKeyFileExt="${pubKeyFileBaseName##*.}"
	privKeyPEMFile="$PULSECERTDIR/$PRIVATESUBDIR/${privKeyFileName}.key"
	pubKeyPEMFile="$PULSECERTDIR/$pubKeyFileName.pem"
	pfxFile="$PULSECERTDIR/$pubKeyFileName.pfx"

	filepath=$(readlink -f "$pubKeyFilePath")
	check_already_installed "$filepath"
        warn_user_for_overwrite "${pubKeyPEMFile}"
	# Public Key Handling 
	if [[ $pubKeyFileExt == *"p7b" || $pubKeyFileExt == *"p7c" ]]; then
		pkcs7Cmd='$OPENSSLCMD pkcs7 -print_certs -in '\"$pubKeyFilePath\"' -out '\"$pubKeyPEMFile\"
		eval $pkcs7Cmd
		check_error $? "$FUNCNAME: convert $pubKeyFileName to PEM format failed"
	else
		# pkcs 8 format should be given as pem/der file here
		if [[ $pubKeyFileExt == *"der" || $pubKeyFileExt == *"cer" ]]; then
			x509Cmd='$OPENSSLCMD x509 -inform der -in '\"$pubKeyFilePath\"' -out '\"$pubKeyPEMFile\"
			eval $x509Cmd
			check_error $? "$FUNCNAME: convert $pubKeyFileName to PEM format failed"
		elif [[ $pubKeyFileExt == *"pem" || $pubKeyFileExt == *"crt" ||
				$pubKeyFileExt == *"key" || $pubKeyFileExt == *"pub" ]]; then
			cp "$pubKeyFilePath" "$pubKeyPEMFile"
		else
			check_error 1 "$FUNCNAME: Unknown Public Key File Format"
		fi
	fi
	# Private Key Handling 
	if [[ $privKeyFileExt == *"der" || $privKeyFileExt == *"cer" ]]; then
		install_keys "der" "$privKeyFilePath" "$privKeyPEMFile"
	elif [[ $privKeyFileExt == *"pem" || $privKeyFileExt == *"crt" ||
				$privKeyFileExt == *"key" ]]; then
		# this command removes the password temporarily to install it in gnome-keyring
		install_keys "pem" "$privKeyFilePath" "$privKeyPEMFile"
	elif [[ $privKeyFileExt == *"pk8" ]]; then
		install_keys "pkcs8" "$privKeyFilePath" "$privKeyPEMFile"
	else
		check_error 1 "$FUNCNAME: Unknown Private Key File Format"
	fi
    echo "Successfully added certificate to Pulse Certificate store."
    cmd='openssl pkcs12 -export -in '\"$pubKeyPEMFile\"' -inkey '\"$privKeyPEMFile\"'  -out '\"$pfxFile\"' -passin pass: -passout pass:'
    eval $cmd
    ret=$?
    if [ $ret == 0 ]; then
	paswd=
    	pk12CMD='${PK12UTIL} -d sql:$HOME/.pki/nssdb  -i '\"${pfxFile}\"'  -W '\"${paswd}\"
    
    	eval $pk12CMD
    	ret=$?
    	if [ $ret != 0 ]; then
        	echo "Failed to add certificate to CEF certificate store."
        else
            echo "Successfully added certificate to CEF Certificate store." 
        fi	
    else
		echo "Failed to convert certificate $2 to pfx format."  
    fi

}
function keyUsage() 
{
    echo "Run command line client Options:"
        echo "    Client Certificate Options:"
        echo "        $SCRNAME install_certificates "
        echo "            [-inpfx < PFX file > ]"  
        echo "            [-inpriv <private file> -inpub <public file>]" 
        echo "        Note: password is required for installing private and public keys separately."
        echo "                                                                                         "
        echo "        $SCRNAME delete_certificates "
        echo "            [-certName <Certificate Name>]"
        echo "        Note: To delete certificates installed in CEF certificate store, run the following command."
        echo "              where, 'Nickname' can be found from '$SCRNAME list_installed_certificates'"
        echo "        /usr/bin/certutil -d sql:${HOME}/.pki/nssdb -D -n <Nickname>"
        echo "        $SCRNAME list_installed_certificates "
        exit 1
}
######################################################################################################
# Function to install certificates
# Args   : certificate details
# Return : None
# function install_certificate ()
function install_certificate()
{
        echo
        echo "Certficate is installing by user: \"$USER\" "\
             "Please make sure that client certificates to be installed by logged in user only."
        read -e -p "Do you want to continue[y/n]: " choice

        if ! [[ "${choice:0:1}" == "Y" || "${choice:0:1}" == "y" ]]; then
            echo "Aborting the certificate installation."
            exit 0;
        fi

	privKeyFileName=""
	pubKeyFileName=""
	echo "install_certificate : $@"
	while [ $# -gt 0 ]
	do
		case "$1" in 
			-inpfx) filename=$(echo "$@" | awk -F '-inpfx|-inpriv|-inpub' '{print $2}'); shift;;
			-inpriv) privKeyFileName=$(echo "$@" | awk -F '-inpfx|-inpriv|-inpub' '{print $2}'); shift;;
			-inpub) pubKeyFileName=$(echo "$@" | awk -F '-inpfx|-inpriv|-inpub' '{print $2}'); shift;;
			-*) keyUsage
		esac
		shift
	done

	#To remove leading and trailing white spaces in filenames. 
	#Cant use space as field separator as folder name itself may contain spaces
	filename="$(echo -e "$filename" | sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//')"
	privKeyFileName="$(echo -e "$privKeyFileName" | sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//')"
	pubKeyFileName="$(echo -e "$pubKeyFileName" | sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//')"

	if [ ! -d $PULSECERTDIR ]; then
		echo "$PULSECERTDIR does not exists. Creating.."
		mkdir -p $PULSECERTDIR
	fi

	if [ ! -d $PULSECERTDIR/$PRIVATESUBDIR ]; then
		echo "$PULSECERTDIR does not exists. Creating.."
		mkdir -p $PULSECERTDIR/$PRIVATESUBDIR
	fi

    if [ ! -d ${HOME}/.pki/nssdb ]; then
        mkdir -p ${HOME}/.pki/nssdb
        ${CERTUTIL} -N -d ${HOME}/.pki/nssdb --empty-password
    fi
   

	if [[ $filename == *".pfx" || $filename == *".p12" ]]
	then
		install_pfx "$filename"
	elif [ "X$privKeyFileName" != "X" ] && [ "X$pubKeyFileName" != "X" ]; then 
		echo "Private Key: $privKeyFileName and Public Key: $pubKeyFileName"
                check_cert_names_same "$privKeyFileName" "$pubKeyFileName"
		install_priv_pub_keys "$privKeyFileName" "$pubKeyFileName"
	else 
		keyUsage
	fi
	#make certificate install directory only for the logged in user.
	chmod -R go-rwx $PULSECERTDIR 
    chmod -R go-rwx ${HOME}/.pki
}
# End of function install_certificate ()
######################################################################################################
######################################################################################################
# Function to delete certificates
# Args   : certificate name 
# Return : None
# function delete_certificate ()
function delete_certificate()
{
	cert_name=""
   	echo "delete_certificate : $@"
	while [ $# -gt 0 ]
	do
		case "$1" in 
			#-certName) cert_name="$2"; shift;;
			-certName) cert_name=$(echo "$@" | awk -F '-certName ' '{print $2}'); shift;;
			-*) keyUsage
		esac
		shift
	done
	if [ "X$cert_name" != "X" ]; then 
		echo "Certificate Name :$cert_name "
		if [ -e $PULSECERTDIR/"$cert_name".pem ]; then 
			rm -rf $PULSECERTDIR/"$cert_name".pem
		else
			echo -e "Public key file $PULSECERTDIR/$cert_name.pem doesn't exists"
		fi
	else
		keyUsage
	fi
}
# End of function delete_certificate ()

#List the installed certificate in pulse certficate store.
function list_certificates()
{
	if ls $PULSECERTDIR/*.pem &>/dev/null
	then
   		for i in $PULSECERTDIR/*.pem;
		do
  			name=$(basename "$i" ".pem")
  			echo -e "\nCertificate Name:" $name;
  			opensslListCmd='$OPENSSLCMD x509 -in '\"$i\"' -text | grep -i "Subject:\|Issuer:\|Validity\|Not Before\|Not After";'
  			eval $opensslListCmd
		done
	else
   		echo "No Certificates found."
	fi
    if [ -d ${HOME}/.pki ]; then
    	echo "###############################################################################"
	echo "Certificates installed for CEF Certificate Store:"
    	${CERTUTIL} -d sql:$HOME/.pki/nssdb -L
    fi
}

######################################################################################################


if [ "X$1" = "Xinstall_certificates" ] ; then
    install_certificate $SCRARGS
elif [ "X$1" = "Xdelete_certificates" ] ; then
    delete_certificate $SCRARGS
elif [ "X$1" = "Xlist_installed_certificates" ] ; then
    list_certificates
else
    keyUsage
fi
